<?php

namespace Utils;

use Google\Auth\Credentials\ServiceAccountCredentials;
use GuzzleHttp\Client;


class FCMNotification
{
    private $credentialsPath;
    private $projectId;
    private $client;
    private $accessToken;

    public function __construct(string $credentialsPath, string $projectId)
    {
        $this->credentialsPath = $credentialsPath;
        $this->projectId = $projectId;
        $this->client = new Client();
        $this->authorize();
    }

    private function authorize()
    {
        $credentials = json_decode(file_get_contents($this->credentialsPath), true);
        $oauth = new ServiceAccountCredentials(
            ['https://www.googleapis.com/auth/firebase.messaging'],
            $credentials
        );
        $tokenData = $oauth->fetchAuthToken();
        if (!isset($tokenData['access_token'])) {
            throw new \Exception('Unable to retrieve access token.');
        }
        $this->accessToken = $tokenData['access_token'];
    }

    public function subscribeToTopic(string $deviceToken, string $topic)
    {
        $url = "https://iid.googleapis.com/iid/v1/{$token}/rel/topics/{$topic}";

        $credentials = json_decode(file_get_contents($this->credentialsPath), true);

        $client = new \GuzzleHttp\Client();
        $client->post($url, [
            'headers' => [
                'Authorization' => 'key=' . $credentials['api_key'],
            ]
        ]);
    }


    public function sendNotification(string $deviceToken, string $title, string $body, $link = [])
    {
        $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";

        $message = [
            "message" => [
                "token" => $deviceToken,
                "notification" => [
                    "title" => $title,
                    "body" => $body
                ]
            ]
        ];

        if (!empty($link)) {
            $message["message"]["webpush"] = [
                "fcm_options" => [
                    "link" => $link
                ]
            ];
        }

        try {
            $response = $this->client->post($url, [
                'headers' => [
                    'Authorization' => "Bearer {$this->accessToken}",
                    'Content-Type' => 'application/json',
                ],
                'json' => $message,
            ]);

            return (string) $response->getBody();
        } catch (\Exception $e) {
            if ($e instanceof \GuzzleHttp\Exception\ClientException) {
                return $e->getResponse()->getBody()->getContents();
            }
            return "Error: " . $e->getMessage();
        }
    }

    public function sendMulticast($title, $body, $data = [], $tokens = [])
    {
        if (empty($tokens)) {
            return false;
        }

        $responses = [];

        foreach($tokens as $token){
            
            $message = [
                "message" => [
                    "token" => $token,
                    "notification" => [
                        "title" => $title,
                        "body" => $body
                    ]
                ]
            ];
            
            $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";

       
            $response = $this->client->post($url, [
                'headers' => [
                        'Authorization' => "Bearer {$this->accessToken}",
                        'Content-Type' => 'application/json',
                ],
                'json' => $message
            ]);
            
            $responses[] = $response->getBody()->getContents();
        }

        
        return $responses;
    }

}
